<?php

namespace App\Http\Controllers\Admin\Webhooks;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\Request;
use App\Events\PaymentProcessed;
use App\Events\PaymentReferrerBonus;
use App\Models\SubscriptionPlan;
use App\Models\PrepaidPlan;
use App\Models\Subscriber;
use App\Models\Payment;
use App\Models\User;
use Illuminate\Support\Facades\Mail;
use App\Mail\PaymentSuccess;
use App\Mail\NewPaymentNotification;
use App\Services\HelperService;
use Exception;

class CoinremitterWebhookController extends Controller
{
    /**
     * Stripe Webhook processing, unless you are familiar with 
     * Stripe's PHP API, we recommend not to modify it
     */
    public function handleCoinremitter(Request $request)
    {
        // Retrieve the raw payload
        $payload = json_decode($request->getContent(), true);
        if (!$payload) {
            Log::info('Invalid Coinremitter payload received.');
            return response()->json(['status' => 'error', 'message' => 'Invalid payload'], 400);
        }

        // Verify the signature
        $signature = $request->header('signature');
        if (!$signature) {
            Log::info('Signature header missing in Coinremitter webhook.');
            return response()->json(['status' => 'error', 'message' => 'Signature missing'], 400);
        }


        // Extract relevant data from the Coinremitter payload
        $invoiceId = $payload['invoice_id'] ?? null;
        $status = $payload['status'] ?? null;
        $customData = json_decode($payload['custom_data1'] ?? '{}', true);

        // Check if the payment is confirmed
        if ($status === 'Paid' && isset($customData['user'])) {
            $user = User::where('id', $customData['user'])->first();

            if (!$user) {
                Log::info("User not found for ID: {$customData['user']}");
                return response()->json(['status' => 'error', 'message' => 'User not found'], 400);
            }

            // Retrieve payment and plan
            $payment = Payment::where('order_id', $invoiceId)->first();
            $plan = ($customData['type'] === 'lifetime')
                ? SubscriptionPlan::where('id', $customData['plan_id'])->first()
                : PrepaidPlan::where('id', $customData['plan_id'])->first();

            if (!$payment || !$plan) {
                Log::info("Payment or plan not found for invoice ID: {$invoiceId}");
                return response()->json(['status' => 'error', 'message' => 'Payment or plan not found'], 400);
            }

            // Handle referral bonus if applicable
            if (config('payment.referral.enabled') === 'on') {
                if (config('payment.referral.payment.policy') === 'first') {
                    if (!Payment::where('user_id', $user->id)->where('status', 'completed')->exists()) {
                        event(new PaymentReferrerBonus($user, $invoiceId, $plan->price, 'Coinremitter'));
                    }
                } else {
                    event(new PaymentReferrerBonus($user, $invoiceId, $plan->price, 'Coinremitter'));
                }
            }

            // Update payment status
            $payment->status = 'completed';
            $payment->save();

            // Update subscriber status
            $subscriber = Subscriber::where('subscription_id', $invoiceId)->first();
            if ($subscriber) {
                $subscriber->status = 'Active';
                $subscriber->save();
            }

            // Update user credits based on plan type
            if ($customData['type'] === 'lifetime') {
                $group = $user->hasRole('admin') ? 'admin' : 'subscriber';
                $user->syncRoles($group);
                $user->group = $group;
                $user->plan_id = $plan->id;
                $user->tokens = $plan->token_credits;
                $user->characters = $plan->characters;
                $user->minutes = $plan->minutes;
                $user->images = $plan->image_credits;
                $user->member_limit = $plan->team_members;
            } else {
                $user->tokens_prepaid += $plan->tokens;
                $user->images_prepaid += $plan->images;
                $user->characters_prepaid += $plan->characters;
                $user->minutes_prepaid += $plan->minutes;
            }
            $user->save();

            // Trigger payment processed event
            event(new PaymentProcessed($user));

            // Send notification emails
            try {
                $admin = User::where('group', 'admin')->first();
                Mail::to($admin)->send(new NewPaymentNotification($payment));
                Mail::to($user)->send(new PaymentSuccess($payment));
            } catch (Exception $e) {
                Log::info('SMTP settings are not setup to send payment notifications via email: ' . $e->getMessage());
            }
        }

        return response()->json(['status' => 'success'], 200);
    }
    
}
